﻿// --------------------------------------------------------------------------
// Project Frocessing
// ActionScript 3.0 drawing library like Processing.
// --------------------------------------------------------------------------
//
// This library is based on Processing.(http://processing.org)
// Copyright (c) 2004-08 Ben Fry and Casey Reas
// Copyright (c) 2001-04 Massachusetts Institute of Technology
// 
// Frocessing drawing library
// Copyright (C) 2008-09  TAKANAWA Tomoaki (http://nutsu.com) and
//					   	  Spark project (www.libspark.org)
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// contact : face(at)nutsu.com
//

package frocessing.core {
	
	import flash.display.Graphics;
	import flash.display.BitmapData;
	import flash.display.Loader;
	import flash.geom.Matrix;
	import flash.net.URLLoader;
	
	import frocessing.color.FColor;
	import frocessing.shape.IFShape;
	import frocessing.text.IFont;
	import frocessing.text.PFontLoader;
	import frocessing.bmp.FImageLoader;
	
	/**
	 * F5Graphics クラスは、Processing の描画メソッドを実装したクラスです.
	 * 
	 * @author nutsu
	 * @version 0.5
	 */
	public class F5Graphics 
	{
		// Color Mode
		public static const RGB        :String = F5C.RGB;
		public static const HSB        :String = F5C.HSB;
		public static const HSV        :String = F5C.HSV;
		
		// Rect Ellipse Image Mode
		public static const CORNER        :int = F5C.CORNER;
		public static const CORNERS       :int = F5C.CORNERS;
		public static const RADIUS        :int = F5C.RADIUS;
		
		public static const CENTER        :int = F5C.CENTER;
		
		// text
		public static const LEFT    	  :int = F5C.LEFT;
		public static const RIGHT   	  :int = F5C.RIGHT;
		public static const BASELINE	  :int = F5C.BASELINE;
		public static const TOP     	  :int = F5C.TOP;
		public static const BOTTOM  	  :int = F5C.BOTTOM;
		
		// Vertex Mode
		public static const POINTS        :int = F5C.POINTS;
		public static const LINES         :int = F5C.LINES; 
		public static const TRIANGLES     :int = F5C.TRIANGLES;
		public static const TRIANGLE_FAN  :int = F5C.TRIANGLE_FAN;
		public static const TRIANGLE_STRIP:int = F5C.TRIANGLE_STRIP;
		public static const QUADS         :int = F5C.QUADS;
		public static const QUAD_STRIP    :int = F5C.QUAD_STRIP;
		
		public static const OPEN      :Boolean = F5C.OPEN;
		public static const CLOSE     :Boolean = F5C.CLOSE;
		
		public static const NORMALIZED    :int = F5C.NORMALIZED;
		public static const IMAGE         :int = F5C.IMAGE;
		
		// PI
		public static const PI         :Number = Math.PI;
		public static const TWO_PI     :Number = Math.PI*2;
		public static const HALF_PI    :Number = Math.PI/2;
		public static const QUART_PI   :Number = Math.PI/4;
		
		//------------------------------
		/**
		 * @private
		 */
		internal var gc:GraphicsEx;
		
		// size -------------------------
		/**
		 * @private
		 */
		internal var _width:uint;
		/**
		 * @private
		 */
		internal var _height:uint;
		
		// color mode -------------------------
		public var colorModeState:String;
		public var colorModeX:Number;
		public var colorModeY:Number;
		public var colorModeZ:Number;
		public var colorModeA:Number;
		
		// calc -------------------------
		/**
		 * @private
		 */
		internal var __calc_color:uint;
		/**
		 * @private
		 */
		internal var __calc_alpha:Number;
		
		// Shape Attributes -------------------------
		/**
		 * @private
		 */
		internal var _rect_mode:int;		//@default CORNER
		/**
		 * @private
		 */
		internal var _ellipse_mode:int;	//@default CENTER
		/**
		 * @private
		 */
		internal var _shape_mode:int;	//@default CORNER
		/**
		 * @private
		 */
		internal var _vertex_mode:int;	//@default 0
		/**
		 * @private
		 */
		internal var _vertex_mode_polygon:Boolean;
		
		// Vertex -------------------------
		/**
		 * @private
		 */
		internal var vertexsX:Array;
		/**
		 * @private
		 */
		internal var vertexsY:Array;
		/**
		 * @private
		 */
		internal var vertexCount:uint;
		/**
		 * @private
		 */
		internal var splineVertexX:Array;
		/**
		 * @private
		 */
		internal var splineVertexY:Array;
		/**
		 * @private
		 */
		internal var splineVertexCount:uint;
		/**
		 * @private
		 */
		internal var vertexsU:Array;
		/**
		 * @private
		 */
		internal var vertexsV:Array;
		
		// Image -------------------------
		/**
		 * @private
		 */
		internal var _image_mode:int;	//@default CORNER
		/**
		 * @private
		 */
		internal var _tint_color:uint;
		/**
		 * @private
		 */
		internal var _tint_do :Boolean;
		/**
		 * @private
		 */
		internal var tintImageCache:TintCache;
		
		// Texture -------------------------
		/**
		 * @private
		 */
		internal var __texture:Boolean;
		/**
		 * @private
		 */
		internal var _texture_mode:int;
		/**
		 * @private
		 */
		internal var _texture_width:int = 1;
		/**
		 * @private
		 */
		internal var _texture_height:int = 1;
		 
		// Style -------------------------
		/**
		 * @private
		 */
		internal var _style_tmp:Array;
		
		
		// Typographics -------------------------
		private var __text_gc:F5Typographics;
		
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 
		 * @param	graphics
		 */
		public function F5Graphics( graphics:Graphics ) 
		{
			__initGC(graphics);
			tintImageCache	= new TintCache();
			__text_gc 		= new F5Typographics( this );
			
			_width = 100;
			_height = 100;
			
			gc.bezierDetail = 20;
			gc.splineDetail = 20;
			gc.splineTightness = 1.0;
			gc.imageSmoothing = false;
			gc.imageDetail	= 4;
			__text_gc.detail  = 1;
			
			__texture = false;
			
			_style_tmp = [];
			defaultSettings();
		}
		
		/**
		 * @private
		 */
		protected function __initGC( graphics:Graphics ):void
		{
			gc = new GraphicsEx(graphics);
		}
		
		/**
		 * default settings
		 * @private
		 */
		protected function defaultSettings():void
		{
			__calc_color	= 0x000000;
			__calc_alpha	= 1.0;
			
			//mode
			colorModeState	= RGB;
			colorModeX		= colorModeY = colorModeZ = 255;
			colorModeA		= 1.0;
			
			//fill
			gc.fillColor	= 0xffffff;
			gc.fillAlpha	= 1.0;
			gc.fillDo		= true;
			
			//stroke
			gc.strokeColor	= 0x000000;
			gc.strokeAlpha  = 1.0;
			gc.thickness    = 0;
			gc.pixelHinting = false;
			gc.scaleMode    = "normal";
			gc.caps         = null;
			gc.joints       = null;
			gc.miterLimit   = 3;
			
			//shape mode
			_rect_mode		= CORNER;
			_ellipse_mode	= CENTER;
			_texture_mode   = NORMALIZED;
			
			//image
			_image_mode     = CORNER;
			_tint_color		= 0xffffffff;
			_tint_do		= false;
			
			//
			gc.applyStroke();
		}
		
		
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * F5Graphics が保持する幅を示します.
		 */
		public function get width():uint { return _width; }
		/**
		 * F5Graphics が保持する高さを示します.
		 */
		public function get height():uint{ return _height; }
		
		/**
		 * 幅と高さを設定します.
		 */
		public function size( width_:uint, height_:uint ):void
		{
			_width  = width_;
			_height = height_;
		}
		
		/**
		 * 描画を開始するときに実行します.
		 * beginDraw時、graphics は clear() されます.
		 */
		public function beginDraw():void {
			clear();
		}
		
		/**
		 * 描画を終了するときに実行します.
		 */
		public function endDraw():void { ; }
		
		//-------------------------------------------------------------------------------------------------------------------
		// DRAW
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 描画されているグラフィックをクリアします.
		 */
		public function clear():void
		{
			gc.clear();
			tintImageCache.dispose();
		}
		
		//------------------------------------------------------------------------------------------------------------------- Path
		
		/**
		 * 現在の描画位置を指定座標に移動します.
		 */
		public function moveTo( x:Number, y:Number, z:Number=0 ):void
		{
			gc.moveTo( x, y );
		}
		
		/**
		 * 現在の描画位置から指定座標まで描画します.
		 */
		public function lineTo( x:Number, y:Number, z:Number=0 ):void
		{
			gc.lineTo( x, y );
		}
		
		/**
		 * 指定されたをコントロールポイント(controlX, controlY) を使用し、現在の描画位置から (anchorX, anchorY)まで2次ベジェ曲線を描画します.
		 */
		public function curveTo( controlX:Number, controlY:Number, anchorX:Number, anchorY:Number ):void
		{
			gc.curveTo( controlX, controlY, anchorX, anchorY );
		}
		/**
		 * @private
		 */
		internal function _curveTo( cx:Number, cy:Number, cz:Number, x:Number, y:Number, z:Number ):void
		{
			curveTo( cx, cy, x, y );
		}
		
		/**
		 * 3次ベジェ曲線を描画します.
		 */
		public function bezierTo( cx0:Number, cy0:Number, cx1:Number, cy1:Number, x:Number, y:Number ):void
		{
			gc.bezierTo( cx0, cy0, cx1, cy1, x, y );
		}
		
		/**
		 * スプライン曲線を描画します.
		 * 
		 * @param	cx0		pre point x
		 * @param	cy0		pre point y
		 * @param	x		target point x
		 * @param	y		target point x
		 * @param	cx1		next point x 
		 * @param	cy1 	next point y
		 */
		public function splineTo( cx0:Number, cy0:Number, x:Number, y:Number, cx1:Number, cy1:Number ):void
		{
			gc.splineTo( cx0, cy0, x, y , cx1, cy1 );
		}
		
		/**
		 * 現在の位置から指定の円弧を描きます.
		 * 
		 * @param	x			center x
		 * @param	y			center y
		 * @param	rx			radius x
		 * @param	ry			radius y
		 * @param	begin		begin radian
		 * @param	end			end radian
		 * @param	rotation
		 */
		public function arcTo( x:Number, y:Number, rx:Number, ry:Number, begin:Number, end:Number, rotation:Number=0 ):void
		{
			if( rotation==0 )
			{
				lineTo( x + rx*Math.cos(begin), y + ry*Math.sin(begin) );
				__arc( x, y, rx, ry, begin, end, rotation );
			}
			else
			{
				var rc:Number = Math.cos(rotation);
				var rs:Number = Math.sin(rotation);
				var xx:Number = rx*Math.cos(begin);
				var yy:Number = ry*Math.sin(begin);
				lineTo( x + xx*rc - yy*rs, y + xx*rs + yy*rc );
				__arc( x, y, rx, ry, begin, end, rotation );
			}
		}
		
		/**
		 * 始点と終点を指定して円弧を描画します.
		 * 
		 * <p>
		 * 始点と終点を指定した円弧には、通常4つの描画候補があります.描画する円弧は、<code>large_arg_flg</code>、<code>sweep_flag</code>により指定されます.
		 * <p>
		 * 
		 * @param	x0					start x
		 * @param	y0					start y
		 * @param	x					end x
		 * @param	y					end y
		 * @param	rx					radius x
		 * @param	ry					radius y
		 * @param	large_arc_flag		大きい方の円弧を描画するかを指定します
		 * @param	sweep_flag			円弧の描画方向の正負を指定します
		 * @param	x_axis_rotation		rotation of ellipse(radian)
		 */
		public function arcCurve( x0:Number, y0:Number, x:Number, y:Number, rx:Number, ry:Number, large_arc_flag:Boolean = false, sweep_flag:Boolean = true, x_axis_rotation:Number = 0 ):void
		{
			moveTo( x0, y0 );
			__arcCurve( x0, y0, x, y, rx, ry, large_arc_flag, sweep_flag, x_axis_rotation );
		}
		
		/**
		 * @private
		 */
		internal function __drawArc( x:Number, y:Number, rx:Number, ry:Number, begin:Number, end:Number, rotation:Number=0 ):void
		{
			if( rotation==0 )
			{
				moveTo( x + rx*Math.cos(begin), y + ry*Math.sin(begin) );
				__arc( x, y, rx, ry, begin, end, rotation );
			}
			else
			{
				var rc:Number = Math.cos(rotation);
				var rs:Number = Math.sin(rotation);
				var xx:Number = rx*Math.cos(begin);
				var yy:Number = ry*Math.sin(begin);
				moveTo( x + xx*rc - yy*rs, y + xx*rs + yy*rc );
				__arc( x, y, rx, ry, begin, end, rotation );
			}
		}
		
		
		/**
		 * @private
		 */
		internal function __arc( x:Number, y:Number, rx:Number, ry:Number, begin:Number, end:Number, rotation:Number=0 ):void
		{
			var segmentNum:int = Math.ceil( Math.abs( 4*(end-begin)/PI ) );
			var delta:Number   = (end - begin)/segmentNum;
			var ca:Number      = 1.0/Math.cos( delta*0.5 );
			var t:Number       = begin;
			var ctrl_t:Number  = begin - delta*0.5;
			var i:int;
			
			if( rotation==0 )
			{
				for( i=1 ; i<=segmentNum ; i++ )
				{
					t += delta;
					ctrl_t += delta;
					curveTo( x + rx*ca*Math.cos(ctrl_t), y + ry*ca*Math.sin(ctrl_t), x + rx*Math.cos(t), y + ry*Math.sin(t) );
				}
			}
			else
			{
				var rc:Number = Math.cos(rotation);
				var rs:Number = Math.sin(rotation);
				var xx:Number;
				var yy:Number;
				var cxx:Number;
				var cyy:Number;
				for( i=1 ; i<=segmentNum ; i++ )
				{
					t += delta;
					ctrl_t += delta;
					xx  = rx*Math.cos(t);
					yy  = ry*Math.sin(t);
					cxx = rx*ca*Math.cos(ctrl_t);
					cyy = ry*ca*Math.sin(ctrl_t);
					curveTo( x + cxx*rc - cyy*rs, y + cxx*rs + cyy*rc , x + xx*rc - yy*rs, y + xx*rs + yy*rc );
				}
			}
		}
		
		/**
		 * @private
		 */
		internal function __arcCurve( x0:Number, y0:Number, x:Number, y:Number, rx:Number, ry:Number, large_arc_flag:Boolean = false, sweep_flag:Boolean = true, x_axis_rotation:Number = 0 ):void
		{			
			var e:Number  = rx/ry;
			var dx:Number = (x - x0)*0.5;
			var dy:Number = (y - y0)*0.5;
			var mx:Number = x0 + dx;
			var my:Number = y0 + dy;
			var rc:Number;
			var rs:Number;
			
			if( x_axis_rotation!=0 )
			{
				rc = Math.cos(-x_axis_rotation);
				rs = Math.sin( -x_axis_rotation);
				var dx_tmp:Number = dx*rc - dy*rs; 
				var dy_tmp:Number = dx*rs + dy*rc;
				dx = dx_tmp;
				dy = dy_tmp;
			}
			
			//transform to circle
			dy *= e;
			
			//
			var len:Number = Math.sqrt( dx*dx + dy*dy );
			var begin:Number;
			
			if( len<rx )
			{
				//center coordinates the arc
				var a:Number  = ( large_arc_flag!=sweep_flag ) ? Math.acos( len/rx ) : -Math.acos( len/rx );
				var ca:Number = Math.tan( a );
				var cx:Number = -dy*ca;
				var cy:Number = dx*ca;
				
				//draw angle
				var mr:Number = Math.PI - 2 * a;
				
				//start angle
				begin = Math.atan2( -dy - cy, -dx - cx );
				
				//deformation back and draw
				cy /= e;
				rc  = Math.cos(x_axis_rotation);
				rs  = Math.sin(x_axis_rotation);
				__arc( mx + cx*rc - cy*rs, my + cx*rs + cy*rc, rx, ry, begin, (sweep_flag) ? begin+mr : begin-(2*Math.PI-mr), x_axis_rotation );
			}
			else
			{
				//half arc
				rx = len;
				ry = rx/e;
				begin = Math.atan2( -dy, -dx );
				__arc( mx, my, rx, ry, begin, (sweep_flag) ? begin+Math.PI : begin-Math.PI, x_axis_rotation );
			}
		}
		
		/**
		 * 描画しているシェイプを閉じます.
		 */
		public function closePath():void
		{
			gc.closePath();
		}
		
		
		public function moveToLast():void
		{
			gc.moveToLast();
		}
		
		/**
		 * 点を描画します.点を描画する色は、線の色が適用されます.
		 */
		public function point( x:Number, y:Number, z:Number=0 ):void
		{
			gc.point( x, y );
		}
		
		//------------------------------------------------------------------------------------------------------------------- 2D Primitives
		
		/**
		 * 現在の線のスタイルを適用し、直線を描画します.
		 */
		public function line( x0:Number, y0:Number, x1:Number, y1:Number ):void
		{
			moveTo( x0, y0 );
			lineTo( x1, y1 );
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、三角形を描画します.
		 */
		public function triangle( x0:Number, y0:Number, x1:Number, y1:Number, x2:Number, y2:Number ):void
		{
			gc.applyFill();
			moveTo( x0, y0 );
			lineTo( x1, y1 );
			lineTo( x2, y2 );
			closePath();
			gc.endFill();
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、四角形を描画します.
		 */
		public function quad( x0:Number, y0:Number, x1:Number, y1:Number, x2:Number, y2:Number, x3:Number, y3:Number ):void
		{
			gc.applyFill();
			moveTo( x0, y0 );
			lineTo( x1, y1 );
			lineTo( x2, y2 );
			lineTo( x3, y3 );
			closePath();
			gc.endFill();
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、円弧を描画します.
		 */
		public function arc( x:Number, y:Number, width_:Number, height_:Number, start_radian:Number, stop_radian:Number ):void
		{
			width_ *= 0.5;
			height_ *= 0.5;
			if ( gc.fillDo )
			{
				gc.applyFill();
				moveTo( x, y );
				lineTo( x + width_*Math.cos(start_radian), y + height_*Math.sin(start_radian) );
				__arc( x, y, width_, height_, start_radian, stop_radian );
				gc.endFill();
			}
			else
			{
				__drawArc( x, y, width_, height_, start_radian, stop_radian );
			}
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、円を描画します.
		 */
		public function circle( x:Number, y:Number, radius:Number ):void
		{
			gc.applyFill();
			__drawArc( x, y, radius, radius, 0, TWO_PI );
			gc.endFill();
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、楕円を描画します.
		 * 
		 * <p>
		 * ellipse()メソッドの引数は、ellipseMode() で指定したモードによりその意味が異なります. モードと引数の関係は以下のようになります.<br/>
		 * デフォルトのモードは、<code>CENTER</code>です.
		 * </p>
		 * 
		 * <ul>
		 * <li>mode CENTER  : ellipse( center x, center y, width, height )</li>
		 * <li>mode CORNERS ： ellipse( left, top, right, bottom )</li>
		 * <li>mode CORNER  : ellipse( left, top, width, height )</li>
		 * <li>mode RADIUS  : ellipse( center x, center y, radius x, radius y )</li>
		 * </ul>
		 */
		public function ellipse( x0:Number, y0:Number, x1:Number, y1:Number ):void
		{
			var w:Number;
			var h:Number;
			switch( _ellipse_mode )
			{
				case CORNERS:
					w  = (x1 - x0)*0.5;
					h  = (y1 - y0)*0.5;
					x0 += w;
					y0 += h;
					break;
				case CORNER:
					w  = x1*0.5;
					h  = y1*0.5;
					x0 += w;
					y0 += h;
					break;
				case RADIUS:
					w  = x1;
					h  = y1;
					break;
				case CENTER:
					w  = x1*0.5;
					h  = y1*0.5;
					break;
			}
			gc.applyFill();
			__drawArc( x0, y0, w, h, 0, TWO_PI );
			gc.endFill();
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、矩形を描画します.
		 * 
		 * <p>
		 * rect()メソッドの引数は、rectMode() で指定したモードによりその意味が異なります. モードと引数の関係は以下のようになります.<br/>
		 * デフォルトのモードは、<code>CORNER</code>です.
		 * </p>
		 * 
		 * <ul>
		 * <li>mode CORNER  : rect( left, top, width, height )</li>
		 * <li>mode CORNERS ： rect( left, top, right, bottom )</li>
		 * <li>mode RADIUS  : rect( center x, center y, radius x, radius y )</li>
		 * <li>mode RADIUS  : rect( center x, center y, width, height )</li>
		 * </ul>
		 */
		public function rect( x0:Number, y0:Number, x1:Number, y1:Number ):void
		{
			var rx:Number;
			var ry:Number;
			switch( _rect_mode )
			{
				case CORNERS:
					break;
				case CORNER:
					x1 += x0;
					y1 += y0;
					break;
				case RADIUS:
					rx = x1;
					ry = y1;
					x1 = x0 + rx;
					y1 = y0 + ry;
					x0 -= rx;
					y0 -= ry;
					break;
				case CENTER:
					rx = x1*0.5;
					ry = y1*0.5;
					x1 = x0 + rx;
					y1 = y0 + ry;
					x0 -= rx;
					y0 -= ry;
					break;
			}
			quad( x0, y0, x1, y0, x1, y1, x0, y1 );
		}
		
		//------------------------------------------------------------------------------------------------------------------- Curves
		
		/**
		 * 現在の塗りと線のスタイルを適用し、3次ベジェ曲線を描画します.
		 */
		public function bezier( x0:Number, y0:Number, cx0:Number, cy0:Number, cx1:Number, cy1:Number, x1:Number, y1:Number ):void
		{
			gc.applyFill();
			moveTo( x0, y0 );
			bezierTo( cx0, cy0, cx1, cy1, x1, y1 );
			gc.endFill();
		}
		
		/**
		 * 現在の塗りと線のスタイルを適用し、スプライン曲線を描画します.
		 */
		public function curve( x0:Number, y0:Number, x1:Number, y1:Number, x2:Number, y2:Number, x3:Number, y3:Number ):void
		{
			beginShape();
			curveVertex( x0, y0 );
			curveVertex( x1, y1 );
			curveVertex( x2, y2 );
			curveVertex( x3, y3 );
			endShape();
		}
		
		/**
		 * 3次ベジェ曲線を描画する際の精度を指定します.デフォルト値は 20 です.
		 * @param	detail_step	指定された数の直線で曲線を近似します
		 */
		public function bezierDetail( detail_step:uint ):void
		{
			gc.bezierDetail = detail_step;
		}
		
		/**
		 * スプライン曲線を描画する際の精度を指定します.デフォルト値は 20 です.
		 * @param	detail_step	指定された数の直線で曲線を近似します
		 */
		public function curveDetail( detail_step:uint ):void
		{
			gc.splineDetail = detail_step;
		}
		
		/**
		 * スプライン曲線の曲率を指定します.デフォルト値は 1.0 です.
		 */
		public function curveTightness( tightness:Number ):void
		{
			gc.splineTightness = tightness;
		}
		
		/**
		 * quadratic bezier function
		 * @param	a	first point on the curve
		 * @param	b	control point
		 * @param	c	second point on the curve
		 * @param	t	value [0,1]
		 */
		public function qbezierPoint( a:Number, b:Number, c:Number, t:Number ):Number
		{
			var tp:Number = 1.0 - t;
			return a*tp*tp + 2*b*t*tp + c*t*t;
		}
		
		/**
		 * cubic bezier function
		 * @param	a	first point on the curve
		 * @param	b	first control point
		 * @param	c	second control point
		 * @param	d	second point on the curve
		 * @param	t	value [0,1]
		 */
		public function bezierPoint( a:Number, b:Number, c:Number, d:Number, t:Number ):Number
		{
			var tp:Number = 1.0 - t;
			return a*tp*tp*tp + 3*b*t*tp*tp + 3*c*t*t*tp + d*t*t*t;
		}
		
		/**
		 * spline function
		 * @param	a	first point on the curve
		 * @param	b	second point on the curve
		 * @param	c	third point on the curve
		 * @param	d	fourth point on the curve
		 * @param	t	value [0,1]
		 */
		public function curvePoint( a:Number, b:Number, c:Number, d:Number, t:Number ):Number
		{
			var v0:Number = gc.splineTightness*( c - a )*0.5;
			var v1:Number = gc.splineTightness*( d - b )*0.5;
			return t*t*t*( 2*b - 2*c + v0 + v1 ) + t*t*( -3*b + 3*c - 2*v0 - v1 ) + v0*t + b;
		}
		
		/**
		 * diff of quadratic bezier function
		 * @param	a	first point on the curve
		 * @param	b	control point
		 * @param	c	second point on the curve
		 * @param	t	value [0,1]
		 */
		public function qbezierTangent( a:Number, b:Number, c:Number, t:Number ):Number
		{
			return 2*( t*( a + c - 2*b ) - a + b );
		}
		
		/**
		 * diff of cubic bezier function
		 * @param	a	first point on the curve
		 * @param	b	first control point
		 * @param	c	second control point
		 * @param	d	second point on the curve
		 * @param	t	value [0,1]
		 */
		public function bezierTangent( a:Number, b:Number, c:Number, d:Number, t:Number ):Number
		{
			return 3*(d-a-3*c+3*b)*t*t + 6*(a+c-2*b)*t - 3*a+3*b;
		}
		
		/**
		 * diff of spline function
		 * @param	a	first control point
		 * @param	b	first point on the curve
		 * @param	c	second point on the curve
		 * @param	d	second control point
		 * @param	t	value [0,1]
		 */
		public function curveTangent( a:Number, b:Number, c:Number, d:Number, t:Number ):Number
		{
			var v0:Number = gc.splineTightness*( c - a )*0.5;
			var v1:Number = gc.splineTightness*( d - b )*0.5;
			return 3*t*t*( 2*b -2*c + v0 + v1) + 2*t*( 3*c - 3*b - v1 - 2*v0 ) + v0;
		}
		
		//------------------------------------------------------------------------------------------------------------------- Vertex
		
		/**
		 * @param	mode	 POINTS, LINES, TRIANGLES, TRIANGLE_FAN, TRIANGLE_STRIP, QUADS, QUAD_STRIP
		 */
		public function beginShape( mode:int=0 ):void
		{
			_vertex_mode    = mode;
			vertexsX      = [];
			vertexsY      = [];
			splineVertexX = [];
			splineVertexY = [];
			vertexsU      = [];
			vertexsV      = [];
			
			vertexCount = 0;
			splineVertexCount = 0;
			
			if ( _vertex_mode_polygon = (mode == 0) )
			{
				gc.applyFill();
			}
		}
		
		/**
		 * 
		 */
		public function endShape( close_path:Boolean=false ):void
		{
			if ( _vertex_mode_polygon )
			{
				if ( close_path )
					gc.closePath();
				endFill();
				_vertex_mode_polygon = false;
			}
			if ( __texture )
			{
				__texture = false;
				gc.endBitmap();
			}
			vertexCount       = 0;
			splineVertexCount = 0;
			_vertex_mode        = 0;
		}
		
		/**
		 * vertex() で 描画する テクスチャ(画像) を設定します.
		 * <p>
		 * texture が適用されるのは、 beginShape() メソッドで以下のモードを指定した場合になります.<br/>
		 * 「　TRIANGLES　TRIANGLE_FAN　TRIANGLE_STRIP　QUADS　QUAD_STRIP 」
		 * </p>
		 * <p>
		 * vertex() メソッドで、　u, v 値を指定する必要があります.
		 * </p>
		 */
		public function texture( textureData:BitmapData ):void
		{
			if ( _tint_do )
				gc.beginBitmap( tintImageCache.getTintImage( textureData, _tint_color ) );
			else
				gc.beginBitmap( textureData );
			__texture = true;
			_texture_width  = textureData.width;
			_texture_height = textureData.height;
		}
		
		/**
		 * @param	mode	NORMALIZED, IMAGE
		 */
		public function textureMode( mode:int ):void
		{
			_texture_mode = mode;
		}
		
		/**
		 * 
		 * @param	x
		 * @param	y
		 * @param	u	texture を指定している場合、u 値を画像の x 座標で指定できます
		 * @param	v	texture を指定している場合、v 値を画像の y 座標で指定できます
		 */
		public function vertex( x:Number, y:Number, u:Number=0, v:Number=0 ):void
		{
			vertexsX[vertexCount] = x;
			vertexsY[vertexCount] = y;
			if ( _texture_mode < 1 )
			{
				u /= _texture_width;
				v /= _texture_height;
			}
			vertexsU[vertexCount] = u;
			vertexsV[vertexCount] = v;
			vertexCount++;
			splineVertexCount = 0;
			
			var t1:uint;
			var t2:uint;
			var t3:uint;
			
			switch( _vertex_mode )
			{
				case POINTS:
					gc.point( x, y );
					break;
				case LINES:
					if ( vertexCount % 2 == 0 )
					{
						t1 = vertexCount - 2;
						gc.moveTo( vertexsX[t1], vertexsY[t1] );
						gc.lineTo( x, y );
					}
					break;
				case TRIANGLES:
					if ( vertexCount % 3 == 0 )
					{
						t1 = vertexCount - 2;
						t2 = vertexCount - 3;
						if ( __texture )
						{
							gc.drawBitmapTriangle( vertexsX[t2], vertexsY[t2], vertexsX[t1], vertexsY[t1], x, y,
												   vertexsU[t2], vertexsV[t2], vertexsU[t1], vertexsV[t1], u, v );
						}
						else
						{
							__vertexTriangle( vertexsX[t2], vertexsY[t2], vertexsX[t1], vertexsY[t1], x, y );
						}
					}
					break;
				case TRIANGLE_FAN:
					if ( vertexCount >= 3 )
					{
						t1 = vertexCount - 2;
						if ( __texture )
						{
							gc.drawBitmapTriangle( vertexsX[0], vertexsY[0], vertexsX[t1], vertexsY[t1], x, y,
												   vertexsU[0], vertexsV[0], vertexsU[t1], vertexsV[t1], u, v );
						}
						else
						{
							__vertexTriangle( vertexsX[0], vertexsY[0], vertexsX[t1], vertexsY[t1], x, y );
						}
					}
					break;
				case TRIANGLE_STRIP:
					if ( vertexCount >= 3 )
					{
						t1 = vertexCount - 2;
						t2 = vertexCount - 3;
						if ( __texture )
						{
							gc.drawBitmapTriangle( x, y, vertexsX[t1], vertexsY[t1], vertexsX[t2], vertexsY[t2],
												   u, v, vertexsU[t1], vertexsV[t1], vertexsU[t2], vertexsV[t2] );
						}
						else
						{
							__vertexTriangle( x, y, vertexsX[t1], vertexsY[t1], vertexsX[t2], vertexsY[t2] );
						}
					}
					break;
				case QUADS:
					if ( vertexCount % 4 == 0 )
					{
						t1 = vertexCount - 2;
						t2 = vertexCount - 3;
						t3 = vertexCount - 4;
						if ( __texture )
						{
							__vertexBitmapQuad( vertexsX[t3], vertexsY[t3], vertexsX[t2], vertexsY[t2], vertexsX[t1], vertexsY[t1], x, y,
											    vertexsU[t3], vertexsV[t3], vertexsU[t2], vertexsV[t2], vertexsU[t1], vertexsV[t1], u, v );
						}
						else
						{
							__vertexQuad( vertexsX[t3], vertexsY[t3], vertexsX[t2], vertexsY[t2], vertexsX[t1], vertexsY[t1], x, y );
						}
					}
					break;
				case QUAD_STRIP:
					if ( vertexCount >= 4 && vertexCount % 2 == 0 )
					{
						t1 = vertexCount - 2;
						t2 = vertexCount - 3;
						t3 = vertexCount - 4;
						if ( __texture )
						{
							__vertexBitmapQuad( vertexsX[t3], vertexsY[t3], vertexsX[t2], vertexsY[t2], x, y, vertexsX[t1], vertexsY[t1],
											    vertexsU[t3], vertexsV[t3], vertexsU[t2], vertexsV[t2], u, v, vertexsU[t1], vertexsV[t1] );
						}
						else
						{
							__vertexQuad( vertexsX[t3], vertexsY[t3], vertexsX[t2], vertexsY[t2], x, y, vertexsX[t1], vertexsY[t1] );
						}
					}
					break;
				default:
					if ( vertexCount > 1 )
						gc.lineTo( x, y );
					else
						gc.moveTo( x, y );
					break;
			}
		}
		
		private function __vertexBitmapQuad( x0:Number, y0:Number, x1:Number, y1:Number, x2:Number, y2:Number, x3:Number, y3:Number,
											 u0:Number, v0:Number, u1:Number, v1:Number, u2:Number, v2:Number, u3:Number, v3:Number ):void
		{
			gc.abortStroke();
			gc.drawBitmapQuad( x0, y0, x1, y1, x2, y2, x3, y3, u0, v0, u1, v1, u2, v2, u3, v3 );
			if ( gc.resumeStroke() )
			{
				gc.moveTo( x0, y0 );
				gc.lineTo( x1, y1 );
				gc.lineTo( x2, y2 );
				gc.lineTo( x3, y3 );
				gc.closePath();
			}
		}
		private function __vertexQuad( x0:Number, y0:Number, x1:Number, y1:Number, x2:Number, y2:Number, x3:Number, y3:Number ):void
		{
			gc.applyFill();
			gc.moveTo( x0, y0 );
			gc.lineTo( x1, y1 );
			gc.lineTo( x2, y2 );
			gc.lineTo( x3, y3 );
			gc.closePath();
			gc.endFill();
		}
		private function __vertexTriangle( x0:Number, y0:Number, x1:Number, y1:Number, x2:Number, y2:Number ):void
		{
			gc.applyFill();
			gc.moveTo( x0, y0 );
			gc.lineTo( x1, y1 );
			gc.lineTo( x2, y2 );
			gc.closePath();
			gc.endFill();
		}
		
		/**
		 * 　
		 */
		public function bezierVertex( cx0:Number, cy0:Number, cx1:Number, cy1:Number, x:Number, y:Number ):void
		{
			if ( _vertex_mode_polygon )
			{
				vertexsX[vertexCount] = x;
				vertexsY[vertexCount] = y;
				vertexCount++;
				splineVertexCount = 0;
				gc.bezierTo( cx0, cy0, cx1, cy1, x, y );
			}
		}
		
		/**
		 *　
		 */
		public function curveVertex( x:Number, y:Number ):void
		{
			if ( _vertex_mode_polygon )
			{
				splineVertexX[splineVertexCount] = x;
				splineVertexY[splineVertexCount] = y;
				splineVertexCount ++;
				
				var t1:int = splineVertexCount - 2;
				var t3:int = splineVertexCount - 4;
				
				if( splineVertexCount>4 )
				{
					gc.splineTo( splineVertexX[t3], splineVertexY[t3],
								 splineVertexX[t1], splineVertexY[t1],
								 x, y );
				}
				else if ( splineVertexCount == 4 )
				{
					var t2:int = splineVertexCount - 3;
					if ( vertexCount > 0 )
					{
						gc.splineTo( splineVertexX[t3], splineVertexY[t3],
								 splineVertexX[t1], splineVertexY[t1],
								 x, y );
					}
					else
					{
						gc.moveTo(splineVertexX[t2], splineVertexY[t2]);
						gc.splineTo( splineVertexX[t3], splineVertexY[t3],
								 splineVertexX[t1], splineVertexY[t1],
								 x, y );
					}
				}
			}
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// SHAPE
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * not implemetned
		 */
		public function shape( s:IFShape, x:Number, y:Number, w:Number = NaN, h:Number = NaN ):void
		{
			;
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// IMAGE
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * Image を読み込みます.
		 * 
		 * <listing>
		 * var f:FImageLoader = loadImage("sample.jpg");
		 * 
		 * if( f.complete )
		 *   image( f.bitmapData, 0, 0 );
		 * </listing>
		 * 
		 * @param	url
		 * @param	loader
		 */
		public function loadImage( url:String, loader:Loader = null ):FImageLoader
		{
			return new FImageLoader( url, null, loader );
		}
		
		
		/**
		 * 画像を描画します.
		 * 
		 * <p>
		 * image()メソッドの引数は、imageMode() で指定したモードによりその意味が異なります. モードと引数の関係は以下のようになります.<br/>
		 * デフォルトのモードは、<code>CORNER</code>です.
		 * </p>
		 * 
		 * <ul>
		 * <li>mode CORNER  : image( left, top, width, height )</li>
		 * <li>mode CORNERS ： image( left, top, right, bottom )</li>
		 * <li>mode CENTER  : image( center x, center y, width, height )</li>
		 * </ul>
		 * 
		 * <p>w、h を省略した場合は、bitmapdata の width と hight が適用されます.</p>
		 */
		public function image( img:BitmapData, x:Number, y:Number, w:Number = NaN, h:Number = NaN ):void
		{
			if ( isNaN(w) || isNaN(h) )
			{
				w = img.width;
				h = img.height;
				if( _image_mode==RADIUS || _image_mode==CENTER )
				{
					x -= w * 0.5;
					y -= h * 0.5;
				}
			}
			else
			{
				switch( _image_mode )
				{
					case CORNERS:
						w  -= x;
						h  -= y;
						break;
					case RADIUS:
						x -= w;
						y -= h;
						w *= 2;
						h *= 2;
						break;
					case CENTER:
						x -= w * 0.5;
						y -= h * 0.5;
						break;
				}
			}
			_image( img, x, y, w, h );
		}
		
		/**
		 * @private
		 */
		internal function _image( img:BitmapData, x:Number, y:Number, w:Number, h:Number, z:Number=0 ):void
		{
			if ( _tint_do )
				gc.beginBitmap( tintImageCache.getTintImage( img, _tint_color ) );
			else
				gc.beginBitmap( img );
			
			gc.drawBitmap( x, y, w, h );
			gc.endBitmap();
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// TEXT
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * font を読み込みます.
		 * @param	font_url	vlw file
		 * @param	loader
		 */
		public function loadFont( font_url:String, loader:URLLoader = null ):PFontLoader
		{
			return new PFontLoader( font_url, loader );
		}
		
		/**
		 * 描画する font を指定します.
		 * @param	font
		 * @param	fontSize
		 */
		public function textFont( font:IFont, fontSize:Number=NaN ):void
		{
			__text_gc.textFont( font, fontSize );
		}
		
		/**
		 * text を描画します.
		 * 
		 * <p>引数の数により色の指定が異なります.</p>
		 * <listing>text( string, x, y )</listing>
		 * <listing>text( string, x, y, z ) when 3D</listing>
		 * <listing>text( string, x, y, width, height )</listing>
		 * <listing>text( string, x, y, width, height, z ) when 3D</listing>
		 */
		public function text( str:String, a:Number, b:Number, c:Number=0, d:Number=0, e:Number=0 ):void
		{
			__text_gc.color = uint( gc.fillAlpha * 0xff ) << 24 | gc.fillColor;
			if ( c > 0 && d > 0 )
				__text_gc.textArea( str, a, b, c, d, e );
			else
				__text_gc.text( str, a, b, c );
		}
		
		/**
		 * 
		 */
		public function textAscent():Number
		{
			return __text_gc.textAscent();
		}
		
		/**
		 * 
		 */
		public function textDescent():Number
		{
			return __text_gc.textDescent();
		}
		
		/**
		 * 文字列の幅を取得します.
		 */
		public function textWidth( str:String ):Number
		{
			return __text_gc.textWidth( str );
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// STYLE
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 現在のスタイルを保持します.
		 * <p>pushStyle() で保持したスタイルは popStyle() で戻されます.</p>
		 * <p>保持されるスタイル</p>
		 * <ul>
		 * <li>colorMode</li>
		 * <li>fill</li>
		 * <li>stroke</li>
		 * <li>stroke attributes(weight,caps,joints...)</li>
		 * <li>rectMode</li>
		 * <li>ellipseMode</li>
		 * <li>imageMode</li>
		 * <li>textFont</li>
		 * <li>textAlign</li>
		 * <li>textSize</li>
		 * <li>textSize</li>
		 * <li>textLeading</li>
		 * <li>tint</li>
		 * </ul>
		 */
		public function pushStyle():void
		{
			var s:FStyle = new FStyle();
			
			s.colorMode 	= colorModeState;
			s.colorModeX 	= colorModeX;
			s.colorModeY 	= colorModeY;
			s.colorModeZ 	= colorModeZ;
			s.colorModeA 	= colorModeA;
			
			s.fillDo 		= gc.fillDo;
			s.fillColor 	= gc.fillColor;
			s.fillAlpha 	= gc.fillAlpha;
			
			s.strokeDo	  	= gc.__stroke;
			s.strokeColor 	= gc.strokeColor;
			s.strokeAlpha 	= gc.strokeAlpha;
			s.thickness 	= gc.thickness;
			s.pixelHinting 	= gc.pixelHinting;
			s.scaleMode 	= gc.scaleMode;
			s.caps 			= gc.caps;
			s.joints 		= gc.joints;
			s.miterLimit 	= gc.miterLimit;
			
			s.rectMode 		= _rect_mode;
			s.ellipseMode 	= _ellipse_mode;
			s.imageMode 	= _image_mode;
			s.shapeMode		= _shape_mode;
			
			s.tintDo 		= _tint_do ;
			s.tintColor 	= _tint_color;
			
			s.textFont		= __text_gc.font;
			s.textAlign 	= __text_gc.align;
			s.textAlignY 	= __text_gc.alignY;
			s.textSize 		= __text_gc.size;
			s.textLeading 	= __text_gc.leading;
			
			_style_tmp.push( s );
		}
		
		/**
		 * 
		 */
		public function popStyle():void
		{
			var s:FStyle = FStyle( _style_tmp.pop() );
			colorModeState	= s.colorMode;
			colorModeX		= s.colorModeX;
			colorModeY		= s.colorModeY;
			colorModeZ 		= s.colorModeZ;
			colorModeA		= s.colorModeA;
			
			gc.fillDo		= s.fillDo;
			gc.fillColor	= s.fillColor;
			gc.fillAlpha	= s.fillAlpha;
			
			gc.strokeColor	= s.strokeColor;
			gc.strokeAlpha  = s.strokeAlpha;
			gc.thickness    = s.thickness;
			gc.pixelHinting = s.pixelHinting;
			gc.scaleMode    = s.scaleMode;
			gc.caps         = s.caps;
			gc.joints       = s.joints;
			gc.miterLimit   = s.miterLimit;
			
			_rect_mode		= s.rectMode;
			_ellipse_mode	= s.ellipseMode;
			_image_mode      = s.imageMode;
			_shape_mode		= s.shapeMode;
			_tint_color		= s.tintColor;
			_tint_do		= s.tintDo;
			
			__text_gc.textFont( s.textFont, s.textSize );
			__text_gc.align	  = s.textAlign;
			__text_gc.alignY  = s.textAlignY;
			__text_gc.leading = s.textLeading;
			
			if ( s.strokeDo )
				gc.applyStroke();
			else
				gc.noStroke();
		}
		
		//------------------------------------------------------------------------------------------------------------------- Shape
		
		/**
		 * @param	mode 	CORNER | CORNERS | RADIUS | CENTER
		 */
		public function rectMode( mode:int ):void
		{
			_rect_mode = mode;
		}
		
		/**
		 * @param	mode 	CORNER | CORNERS | RADIUS | CENTER
		 */
		public function ellipseMode( mode:int ):void
		{
			_ellipse_mode = mode;
		}
		
		/**
		 *  @param	mode 	CORNER | CORNERS | CENTER
		 */
		public function imageMode( mode:int ):void
		{
			_image_mode = mode;
		}
		
		/**
		 *  @param	mode 	CORNER | CORNERS | CENTER
		 */
		public function shapeMode( mode:int ):void
		{
			_shape_mode = mode;
		}
		
		/**
		 * 画像を描画する場合の Smoothing を設定します.
		 */
		public function imageSmoothing( smooth:Boolean ):void
		{
			gc.imageSmoothing = smooth;
		}
		
		/**
		 * 画像を変形して描画する際の精度を指定します.
		 */
		public function imageDetail( segmentNumber:uint ):void
		{
			gc.imageDetail = segmentNumber;
		}
		
		/**
		 * text() で PFont　を描画する際の画像精度.
		 */
		public function fontImageDetail( segmengNumber:uint ):void
		{
			__text_gc.detail = segmengNumber;
		}
		
		//------------------------------------------------------------------------------------------------------------------- Text
		
		/**
		 * text の　size を指定します.
		 * @param	fontSize
		 */
		public function textSize( fontSize:Number ):void
		{
			__text_gc.size = fontSize;
		}
		
		/**
		 * text の align を指定します.
		 * @param	align	CENTER,LEFT,RIGHT
		 * @param	alignY	BASELINE,TOP,BOTTOM
		 */
		public function textAlign( align:int, alignY:int = 0 ):void
		{
			__text_gc.align  = align;
			__text_gc.alignY = alignY;
		}
		
		/**
		 * text の　行高 を指定します.
		 */
		public function textLeading( leading:Number ):void
		{
			__text_gc.leading = leading;
		}
		
		/**
		 * not implemented
		 */
		public function textMode( mode:int ):void
		{
			;
		}
		
		//------------------------------------------------------------------------------------------------------------------- BACKGROUND
		
		/**
		 * 背景を描画します.このメソッドを実行すると、現在の描画内容がクリアされます.
		 * 
		 * <p>引数の数により色の指定が異なります.</p>
		 * 
		 * <listing>background( gray )</listing>
		 * <listing>background( gray, alpha )</listing>
		 * <listing>background( hex )</listing>
		 * <listing>background( hex, alpha )</listing>
		 * <listing>background( red, green, blue )</listing>
		 * <listing>background( red, green, blue, alpha )</listing>
		 * <listing>background( hue, saturation, brightness )</listing>
		 * <listing>background( hue, saturation, brightness, alpha )</listing>
		 */
		public function background( c1:Number, c2:Number=NaN, c3:Number=NaN, c4:Number=NaN ):void
		{
			if ( _width > 0 && _height > 0 )
			{
				__calcColor( c1, c2, c3, c4 );
				gc.__BG( _width, _height, __calc_color, __calc_alpha );
			}
		}
		
		//------------------------------------------------------------------------------------------------------------------- FILL
		
		/**
		 * 塗りの色、透明度を指定します.
		 * 
		 * <p>引数の数により色の指定が異なります.</p>
		 * 
		 * <listing>fill( gray )</listing>
		 * <listing>fill( gray, alpha )</listing>
		 * <listing>fill( hex )</listing>
		 * <listing>fill( hex, alpha )</listing>
		 * <listing>fill( red, green, blue )</listing>
		 * <listing>fill( red, green, blue, alpha )</listing>
		 * <listing>fill( hue, saturation, brightness )</listing>
		 * <listing>fill( hue, saturation, brightness, alpha )</listing>
		 */
		public function fill( c1:Number, c2:Number=NaN, c3:Number=NaN, c4:Number=NaN ):void
		{
			__calcColor( c1, c2, c3, c4 );
			gc.fillColor = __calc_color;
			gc.fillAlpha = __calc_alpha;
			gc.fillDo = true;
		}
		
		/**
		 * 塗りが描画されないようにします.
		 */
		public function noFill():void
		{
			gc.fillDo = false;
		}
		
		//------------------------------------------------------------------------------------------------------------------- STROKE
		
		/**
		 * 線の色、透明度を指定します.
		 * このメソッドにより lineStyle　が実行され線のスタイルが適用されます.
		 * 
		 * <p>引数の数により色の指定が異なります.</p>
		 * 
		 * <listing>stroke( gray )</listing>
		 * <listing>stroke( gray, alpha )</listing>
		 * <listing>stroke( hex )</listing>
		 * <listing>stroke( hex, alpha )</listing>
		 * <listing>stroke( red, green, blue )</listing>
		 * <listing>stroke( red, green, blue, alpha )</listing>
		 * <listing>stroke( hue, saturation, brightness )</listing>
		 * <listing>stroke( hue, saturation, brightness, alpha )</listing>
		 */
		public function stroke( c1:Number, c2:Number=NaN, c3:Number=NaN, c4:Number=NaN ):void
		{
			__calcColor( c1, c2, c3, c4 );
			gc.strokeColor = __calc_color;
			gc.strokeAlpha = __calc_alpha;
			gc.applyStroke();
		}
		
		/**
		 * 線が描画されないようにします.
		 */
		public function noStroke():void
		{
			gc.noStroke();
		}
		
		/**
		 * 線の太さを指定します.有効な値は 0～255 です.
		 */
		public function strokeWeight( thickness:Number ):void
		{
			gc.thickness = thickness;
			gc.reapplyStroke();
		}
		
		/**
		 * 線の角で使用する接合点の外観の種類を指定します.
		 * @see	flash.display.JointStyle
		 */
		public function strokeJoin( jointStyle:String ):void
		{
			gc.joints = jointStyle;
			gc.reapplyStroke();
		}
		
		/**
		 * 線の終端のキャップの種類を指定します.
		 * @see	flash.display.CapsStyle
		 */
		public function strokeCap( capsStyle:String ):void
		{
			gc.caps = capsStyle;
			gc.reapplyStroke();
		}
		
		/**
		 * 線をヒンティングするかどうかを示します.
		 */
		public function strokePixelHint( pixelHinting:Boolean ):void
		{
			gc.pixelHinting = pixelHinting;
			gc.reapplyStroke();
		}
		
		/**
		 * 使用する拡大 / 縮小モードを指定する LineScaleMode クラスの値を示します.
		 * @see flash.display.LineScaleMode
		 */
		public function strokeScaleMode( scaleMode:String ):void
		{ 
			gc.scaleMode = scaleMode;
			gc.reapplyStroke();
		}
		
		/**
		 * マイターが切り取られる限度を示す数値を示します.
		 */
		public function strokeMiterLimit( miterLimit:Number ):void
		{ 
			gc.miterLimit = miterLimit;
			gc.reapplyStroke();
		}
		
		//------------------------------------------------------------------------------------------------------------------- TINT
		
		/**
		 * Tint Color を指定します.
		 * 
		 * <p>引数の数により色の指定が異なります.</p>
		 * 
		 * <listing>tint( gray )</listing>
		 * <listing>tint( gray, alpha )</listing>
		 * <listing>tint( hex )</listing>
		 * <listing>tint( hex, alpha )</listing>
		 * <listing>tint( red, green, blue )</listing>
		 * <listing>tint( red, green, blue, alpha )</listing>
		 * <listing>tint( hue, saturation, brightness )</listing>
		 * <listing>tint( hue, saturation, brightness, alpha )</listing>
		 */
		public function tint( c1:Number, c2:Number=NaN, c3:Number=NaN, c4:Number=NaN ):void
		{
			__calcColor( c1, c2, c3, c4 );
			_tint_color = FColor.toARGB( __calc_color, __calc_alpha );
			_tint_do    = ( _tint_color != 0xffffffff );
		}
		
		/**
		 * Tint を無効にします.
		 */
		public function noTint():void
		{
			_tint_color = 0xffffffff;
			_tint_do    = false;
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		
		public function get fillColor():uint{ return gc.fillColor; }
		public function set fillColor( value:uint ):void
		{
			gc.fillColor = value;
			gc.fillDo = true;
		}
		
		public function get fillAlpha():Number{ return gc.fillAlpha; }
		public function set fillAlpha( value:Number ):void
		{
			gc.fillAlpha = value;
			gc.fillDo = true;
		}
		
		public function get strokeColor():uint{ return gc.strokeColor; }
		public function set strokeColor( value:uint ):void
		{
			gc.strokeColor = value;
			gc.applyStroke();
		}
		
		public function get strokeAlpha():Number{ return gc.strokeAlpha; }
		public function set strokeAlpha( value:Number ):void
		{
			gc.strokeAlpha = value;
			gc.applyStroke();
		}
		
		/**
		 * Tint Color を 32bit Color で示します.
		 */
		public function get tintColor():uint{ return _tint_color; }
		public function set tintColor( value:uint ):void
		{
			_tint_color = value;
			_tint_do    = ( _tint_color != 0xffffffff );
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// GRAPHICS
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * 指定されている線のスタイルをを適用します.
		 */
		public function applyStroke():void
		{
			gc.applyStroke();
		}
		
		/**
		 * 指定されている塗りで beginFill() を実行します.
		 */
		public function applyFill():void
		{
			gc.applyFill();
		}
		
		/**
		 * 線のスタイルを指定します.
		 */
		public function lineStyle( thickness:Number = NaN, color:uint = 0, alpha:Number = 1,
								   pixelHinting:Boolean = false, scaleMode:String = "normal",
								   caps:String=null,joints:String=null,miterLimit:Number=3):void
		{
			gc.lineStyle( thickness, color, alpha, pixelHinting, scaleMode, caps, joints, miterLimit );
		}
		
		/**
		 * 線スタイルのグラデーションを指定します.
		 */
		public function lineGradientStyle( type:String, colors:Array, alphas:Array, ratios:Array,
										   matrix:Matrix = null, spreadMethod:String = "pad", interpolationMethod:String = "rgb",
										   focalPointRatio:Number=0.0 ):void
		{
			gc.lineGradientStyle( type, colors, alphas, ratios, matrix, spreadMethod, interpolationMethod, focalPointRatio );
		}
		
		/**
		 * 今後の描画に使用する単色塗りを指定します.
		 */
		public function beginFill( color:uint, alpha:Number=1.0 ):void
		{
			gc.beginFill( color, alpha );
		}
		
		/**
		 * 描画領域をビットマップイメージで塗りつぶします.
		 */
		public function beginBitmapFill(bitmap:BitmapData,matrix:Matrix=null,repeat:Boolean=true,smooth:Boolean=false):void
		{
			gc.beginBitmapFill( bitmap, matrix, repeat, smooth );
		}
		
		/**
		 * 今後の描画に使用するグラデーション塗りを指定します.
		 */
		public function beginGradientFill(type:String, color:Array, alphas:Array, ratios:Array, matrix:Matrix=null, spreadMethod:String="pad", interpolationMethod:String="rgb",focalPointRation:Number=0.0):void
		{
			gc.beginGradientFill( type, color, alphas, ratios, matrix, spreadMethod, interpolationMethod, focalPointRation );
		}
		
		/**
		 * beginFill()、beginGradientFill()、または beginBitmapFill() メソッドへの最後の呼び出し以降に追加された線と曲線に塗りを適用します.
		 */
		public function endFill():void
		{
			gc.endFill();
		}
		
		//-------------------------------------------------------------------------------------------------------------------
		// Color Mode
		//-------------------------------------------------------------------------------------------------------------------
		
		/**
		 * カラーモードと、色の有効値を設定します.
		 * 
		 * <p>引数の数により有効値の設定が異なります.</p>
		 * 
		 * <listing>colorMode( RGB, value);</listing>
		 * <p>range1～range4 全てに value が適用されます.</p>
		 *  
		 * <listing>colorMode( RGB, value1, value2, value3 );</listing>
		 * <p>colorModeX に value1、colorModeY に value2、colorModeZ に value3 が適用されます.colorModeAは変更されません.</p>
		 * 
		 * <listing>colorMode( RGB, value1, value2, value3, value4 );</listing>
		 * <p>colorModeX～colorModeA　をそれぞれ個別に指定します.</p>
		 * 
		 * @param	mode	RGB,HSB,HSV
		 * @param	range1	colorModeX
		 * @param	range2	colorModeY
		 * @param	range3	colorModeZ
		 * @param	range4	colorModeA
		 */
		public function colorMode( mode:String, range1:Number=0xff, range2:Number=NaN, range3:Number=NaN, range4:Number=NaN ):void
		{
			colorModeState = mode;
			if ( range4 > 0 )
			{
				colorModeX = range1;
				colorModeY = range2;
				colorModeZ = range3;
				colorModeA = range4;
			}
			else if ( range3 > 0 )
			{
				colorModeX = range1;
				colorModeY = range2;
				colorModeZ = range3;
			}
			else
			{
				colorModeX = colorModeY = colorModeZ = colorModeA = range1;
			}
		}
		
		/**
		 * color uint を取得します.
		 * 
		 * <p>引数の数により色の指定が異なります.</p>
		 * 
		 * <listing>color( gray )</listing>
		 * <listing>color( gray, alpha )</listing>
		 * <listing>color( hex )</listing>
		 * <listing>color( hex, alpha )</listing>
		 * <listing>color( red, green, blue )</listing>
		 * <listing>color( red, green, blue, alpha )</listing>
		 * <listing>color( hue, saturation, brightness )</listing>
		 * <listing>color( hue, saturation, brightness, alpha )</listing>
		 * 
		 * @param	c1	first value
		 * @param	c2	second value
		 * @param	c3	third value
		 * @param	c4	fourth value
		 */
		public function color( c1:Number, c2:Number=NaN, c3:Number=NaN, c4:Number=NaN ):uint
		{
			if ( c4 >= 0 )
			{
				if ( colorModeState == HSB )
					return FColor.HSVtoValue( 360*c1/colorModeX , c2/colorModeY, c3/colorModeZ, c4/colorModeA );
				else
					return FColor.RGBtoValue( uint(c1/colorModeX*0xff), uint(c2/colorModeY*0xff), uint(c3/colorModeZ*0xff), c4/colorModeA );
			}
			else if ( c3 >= 0 )
			{
				if ( colorModeState == HSB )
					return FColor.HSVtoValue( 360*c1/colorModeX , c2/colorModeY, c3/colorModeZ );
				else
					return FColor.RGBtoValue( uint(c1/colorModeX*0xff), uint(c2/colorModeY*0xff), uint(c3/colorModeZ*0xff) );
			}
			else if ( c2 >= 0 )
			{
				if( c1<=255 )
					return FColor.GrayToValue( uint(c1/colorModeX*0xff), c2/colorModeA );
				else
					return FColor.toARGB( uint(c1), c2 / colorModeA );	
			}
			else
			{
				if( c1<=255 )
					return FColor.GrayToValue( uint(c1/colorModeX*0xff) );
				else
					return uint(c1);
			}
		}
		
		/**
		 * @private
		 */
		internal function __calcColor( c1:Number, c2:Number=NaN, c3:Number=NaN, c4:Number=NaN ):void
		{
			if ( c4 >= 0 )
			{
				__calc_alpha = c4 / colorModeA;
				if ( colorModeState == HSB )
					__calc_color = FColor.HSVtoValue( 360*c1/colorModeX , c2/colorModeY, c3/colorModeZ );
				else
					__calc_color = FColor.RGBtoValue( uint(c1/colorModeX*0xff), uint(c2/colorModeY*0xff), uint(c3/colorModeZ*0xff) );
			}
			else if ( c3 >= 0 )
			{
				__calc_alpha = 1.0;
				if ( colorModeState == HSB )
					__calc_color = FColor.HSVtoValue( 360*c1/colorModeX , c2/colorModeY, c3/colorModeZ );
				else
					__calc_color = FColor.RGBtoValue( uint(c1/colorModeX*0xff), uint(c2/colorModeY*0xff), uint(c3/colorModeZ*0xff) );
			}
			else if ( c2 >= 0 )
			{
				__calc_alpha = c2 / colorModeA;
				if( c1<=0xff )
					__calc_color = FColor.GrayToValue( uint(c1 / colorModeX * 0xff) );
				else if ( c1 >>> 24 > 0 )
					__calc_color = c1 & 0xffffff;
				else
					__calc_color = uint(c1);
			}
			else
			{
				__calc_alpha = 1.0;
				if ( c1 <= 0xff )
				{
					__calc_color = FColor.GrayToValue( uint(c1 / colorModeX * 0xff) );
				}
				else if ( c1 >>> 24 > 0 )
				{
					__calc_color = c1 & 0xffffff;
					__calc_alpha = (c1 >>> 24) / 0xff;
				}
				else
				{
					__calc_color = uint(c1);
				}
			}
		}
		
	}
	
}

//-------------------------------------------------------------------------------------------------------------------

import flash.display.BitmapData;
import flash.utils.Dictionary;
import flash.geom.ColorTransform;

class TintCache 
{
	private var d:Dictionary;
	private var ct:ColorTransform;
	
	public function TintCache()
	{
		d  = new Dictionary(false);
		ct = new ColorTransform();
	}
	
	/**
	 * カラー調整した BitmapData を取得.
	 * 指定の TintColor に対応した BitmapData が存在しない場合、カラー調整したデータを生成.
	 */
	public function getTintImage( src:BitmapData, color32:uint ):BitmapData
	{
		if ( color32 != 0xffffffff )
		{
			var tint_img:BitmapData;
			if ( d[src] == null )
			{
				d[src] = [];
				d[src][color32] = tint_img = tint( src, color32 );
			}
			else
			{
				tint_img = d[src][color32];
				if ( tint_img == null )
					d[src][color32] = tint_img = tint( src, color32 );
			}
			return tint_img;
		}
		else
		{
			return src;
		}
	}
	
	private function tint( src:BitmapData, color32:uint ):BitmapData
	{
		var img:BitmapData = src.clone();
		ct.alphaMultiplier = ( color32>>>24 ) / 0xff;
		ct.redMultiplier   = ( color32>>16 & 0xff ) / 0xff;
		ct.greenMultiplier = ( color32>>8 & 0xff ) / 0xff;
		ct.blueMultiplier  = ( color32 & 0xff ) / 0xff;
		img.colorTransform( img.rect, ct );
		return img;
	}
	
	/**
	 * 保持しているカラー調整済み BitmapData を dispose.
	 * nullを指定した場合、全てのデータを dispose
	 */
	public function dispose( targetSrc:BitmapData=null ):void
	{
		var d2:Array;
		var tintImg:BitmapData;
		if ( targetSrc == null )
		{
			for ( var img:* in d )
			{
				d2 = d[img];
				for each ( tintImg in d2 )
				{
					tintImg.dispose();
				}
				delete d[img];
			}
		}
		else
		{
			d2 = d[targetSrc];
			if ( d2 != null )
			{
				for each ( tintImg in d2 )
				{
					tintImg.dispose();
				}
				delete d[targetSrc];
			}
		}
	}
}
